

#include "MeetingSiteStorager.h"
#include "jcfcoreutils/StringUtils.h"
#include "jcfcoreutils/TimeUtils.h"
#include "JabberMeetingAccountWapiSync.h"
#include "JabberMeetingAccountCupSync.h"
#include "MeetingAccountControllerImpl.h"
#include "MeetingAccountControllerImpl.h"
#include "MeetingAccountSDKWrapMac.h"
#include "MeetingAccountSDKWrapWin.h"
#include "../JabberServiceProvider.h"
#include "csfunified/framework/UnifiedFactory.h"
#include "csfunified/framework/ServicesDispatcher.h"
#include "csfunified/framework/FunctorTask.h"

#ifndef WIN32
#include <mach/mach_time.h>
#endif


namespace CSFUnified
{
    static CSFLogger* jcfMeetingLogger = CSFLogger_getLogger("MeetingService-MeetingSiteStorager");
    
    MeetingSiteStorager::MeetingSiteStorager() : m_wapiSync(this), m_cupSync(this), m_accountSync(&m_cupSync)
    {
        reset();

		newTimer::getInstant()->startTimer(this, 1000 * 60);
    }
    
    MeetingSiteStorager::~MeetingSiteStorager()
    {
        reset();

		if (newTimer::getInstant()->isRunning())
		{
			newTimer::getInstant()->stopTimer();
		}
    }
    
    MeetingSiteStorager * MeetingSiteStorager::getInstance()
    {
        static MeetingSiteStorager * s_storager = NULL;
        if (NULL == s_storager)
        {
            s_storager = new MeetingSiteStorager();
        }
        
        return s_storager;
    }
    
    void MeetingSiteStorager::onTimer()
    {
        if (m_defaultSiteUrl.length() == 0 || !m_isActive || m_sessionTicket.length() == 0)
        {
            CSFLogDebugS(jcfMeetingLogger, "MeetingSiteStorager::onTimer() defaultSiteUrl = " << m_defaultSiteUrl
                         << "    isActive = " << m_isActive
                         << "    session ticket length = " << m_sessionTicket.length());
            return;
        }
        
        long tickCount = getLocalTickCount();
        
        if (tickCount < m_sessionTicketTimeout)
        {
            CSFLogDebugS(jcfMeetingLogger, "MeetingSiteStorager::onTimer() sessionTicketTimeout = " << m_sessionTicketTimeout << "    tickCount = " << tickCount);
            return;
        }
        
        csf::ScopedReadRWLock lock(mLock);
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = JabberServiceProvider::getInstance().getUnifiedFactory()->getServicesDispatcher();
        if (pServiceDispatcher != NULL)
        {
            pServiceDispatcher->enqueueBlock(boost::bind(&MeetingSiteStorager::refreshSessionTicket, this), "MeetingSiteStorager::refreshSessionTicket");
        }
    }
    
    void MeetingSiteStorager::refreshSessionTicket()
    {
        CSFLogDebugS(jcfMeetingLogger, "MeetingSiteStorager::refreshSessionTicket()");
		m_isRefreshSK = true;
        m_accountSync->getDefaultSite();
    }
    
    
    void MeetingSiteStorager::onGetAllSite(bool isSuccess, std::list<SMART_PTR_NS::shared_ptr<MeetingSiteImpl> >& siteList)
    {
        CSFLogDebugS(jcfMeetingLogger, "MeetingSiteStorager::onGetAllSite() isSuccess: " << isSuccess);
        
        std::list<std::string> siteUrlList;
        
        std::list<SMART_PTR_NS::shared_ptr<MeetingSiteImpl> >::iterator it = siteList.begin();
        while (it != siteList.end())
        {
            m_mapAccountInfo[(*it)->getSiteUrl()] = (*it);
            
            if ((*it)->getSSOFlag() == 2)
            {
                siteUrlList.push_back((*it)->getSiteUrl());
            }
            
            it++;
        }
        
        if (m_mapAccountInfo.size() > 0)
        {
            if (siteUrlList.size() > 0)
            {
                MeetingAccountSDKWrap::getInstance()->checkSSOSiteList(siteUrlList);
            }
            else
            {
				m_isRefreshSK = false;
                m_accountSync->getDefaultSite();
            }
        }
        else
        {
            setReady();
            m_defaultSiteUrl = "";
            m_isActive = false;

			#ifdef __APPLE__
			// notify UI stop wait.
            SMART_PTR_NS::shared_ptr<MeetingSiteImpl> site;
            MeetingAccountControllerImpl::getInstance()->fire_onMeetingSiteActived(false, "", site);
			#endif
        }
    }
    
	void MeetingSiteStorager::onGetDefaultSite(JABBER_MEETING_ACCOUNT_ERROR_CODE errorCode, std::string errorMsg, SMART_PTR_NS::shared_ptr<MeetingSiteImpl> site, std::string sessionTicket, long keepAliveTime)
    {
        std::string siteUrl = site.get() == NULL? "" : site->getSiteUrl();
        CSFLogDebugS(jcfMeetingLogger, "MeetingSiteStorager::onGetDefaultSite() site: " << siteUrl << "    errorCode: " << (int)errorCode << "    errorMessage: " << errorMsg);

		if (m_isRefreshSK && errorCode != e_jmaec_success)
		{
			m_isRefreshSK = false;
			CSFLogDebugS(jcfMeetingLogger, "MeetingSiteStorager::onGetDefaultSite() refresh SDK failed will refresh again after one minute");
			return;
		}

        if (NULL != site.get())
        {
            m_defaultSiteUrl = site->getSiteUrl();
        }
        
        if (errorCode != e_jmaec_success || NULL == site.get())
        {
            setReady();
            m_isActive = false;
			setLastError(errorCode, errorMsg);
			MeetingAccountControllerImpl::getInstance()->fire_onMeetingSiteActived(false, "", site);
            return;
        }

        SMART_PTR_NS::shared_ptr<MeetingSite> siteEx = getDefaultSite();
		int ssoFlag = 0;
		if (NULL != siteEx.get())
		{
			ssoFlag = siteEx->getSSOFlag();
		}

        m_inProcessMeetingSite = site;
        m_inProcessSessionTicket = sessionTicket;
        m_inProcessKeepAliveTime = keepAliveTime;
        
        if (sessionTicket.length() > 0)
        {
            MeetingAccountSDKWrap::getInstance()->verifyAccountWithSessionTicket(site->getSiteUrl(), site->getUserName(), sessionTicket);
        }
        else
        {
			if (1 == ssoFlag)
            {
                MeetingAccountControllerImpl::getInstance()->fire_requireRefreshSessionTicket(m_defaultSiteUrl, false);
            }
            else
            {
                MeetingAccountSDKWrap::getInstance()->verifyAccountWithPassword(site->getSiteUrl(), site->getUserName(), site->getPassword());
            }
        }
    }
    
	void MeetingSiteStorager::onSetDefaultSite(JABBER_MEETING_ACCOUNT_ERROR_CODE errorCode, std::string errorMsg, SMART_PTR_NS::shared_ptr<MeetingSiteImpl> site, std::string sessionTicket, long keepAliveTime)
    {
        std::string siteUrl = site.get() == NULL? "" : site->getSiteUrl();
        
        CSFLogDebugS(jcfMeetingLogger, "MeetingSiteStorager::onSetDefaultSite() site: " << siteUrl << "    errorCode: " << (int)errorCode << "    errorMessage: " << errorMsg);
        
        std::string originalSiteUrl = (NULL != m_inProcessMeetingSite.get()) ? m_inProcessMeetingSite->getSiteUrl() : "";
		if (errorCode == e_jmaec_error_setup_error)
		{
			setLastError(0, errorMsg);
			MeetingAccountControllerImpl::getInstance()->fire_onMeetingSiteActived(false, originalSiteUrl, m_inProcessMeetingSite);
		}
		else if (errorCode != e_jmaec_success || NULL == site.get())
        {
			setLastError(errorCode, errorMsg);
			MeetingAccountControllerImpl::getInstance()->fire_onMeetingSiteActived(false, originalSiteUrl, m_inProcessMeetingSite);
            return;
        }
        
        m_inProcessMeetingSite = site;
        m_inProcessSessionTicket = sessionTicket;
        m_inProcessKeepAliveTime = keepAliveTime;

        if (sessionTicket.length() > 0)
        {
            MeetingAccountSDKWrap::getInstance()->verifyAccountWithSessionTicket(site->getSiteUrl(), site->getUserName(), sessionTicket);
        }
        else
        {
            MeetingAccountSDKWrap::getInstance()->verifyAccountWithPassword(site->getSiteUrl(), site->getUserName(), site->getPassword());
        }
    }
    
    void MeetingSiteStorager::start()
    {
        CSFLogDebugS(jcfMeetingLogger, "MeetingSiteStorager::start()");
        
        reset();
        
        if (JabberServiceProvider::getInstance().isCupMode())
        {
            m_accountSync = &m_cupSync;
        }
        else
        {
            m_accountSync = &m_wapiSync;
        }
        
		MeetingAccountSDKWrap::getInstance()->start();
        m_accountSync->start();
        m_accountSync->syncAllSite();
    }
    
    void MeetingSiteStorager::stop()
    {
        CSFLogDebugS(jcfMeetingLogger, "MeetingSiteStorager::stop()");
        
        m_accountSync->stop();
        MeetingAccountSDKWrap::getInstance()->stop();
        reset();
    }
    
    bool MeetingSiteStorager::isActive()
    {
        return m_isActive;
    }
    
    
    bool MeetingSiteStorager::isReady()
    {
        return m_isReady;
    }
    
    void MeetingSiteStorager::setReady()
    {
        if (!m_isReady)
        {
            m_isReady = true;
        }
    }

    
    int MeetingSiteStorager::getCount()
    {
        return m_mapAccountInfo.size();
    }
    
    SMART_PTR_NS::shared_ptr<MeetingSite> MeetingSiteStorager::getDefaultSite()
    {
        SMART_PTR_NS::shared_ptr<MeetingSiteImpl> meetingSite;
        std::map<std::string, SMART_PTR_NS::shared_ptr<MeetingSiteImpl> >::iterator it = m_mapAccountInfo.find(m_defaultSiteUrl);
        if (it != m_mapAccountInfo.end())
        {
            meetingSite.reset(new MeetingSiteImpl(it->second.get()));
        }
        
        return meetingSite;
    }
    
    SMART_PTR_NS::shared_ptr<MeetingSite> MeetingSiteStorager::getSiteByIndex(int index)
    {
        int i = 0;
        SMART_PTR_NS::shared_ptr<MeetingSiteImpl> meetingSite;
        std::map<std::string, SMART_PTR_NS::shared_ptr<MeetingSiteImpl> >::iterator it = m_mapAccountInfo.begin();
        while (it != m_mapAccountInfo.end())
        {
            if (i == index)
            {
                meetingSite.reset(new MeetingSiteImpl(it->second.get()));
                break;
            }
            
            i++;
            it++;
        }
        
        if (NULL != meetingSite.get())
        {
            bool isDefault = (meetingSite->getSiteUrl() == m_defaultSiteUrl) ? true : false;
            meetingSite->setDefaultSiteFlag(isDefault);
            meetingSite->setActiveFlag(isDefault && m_isActive);
        }
        return meetingSite;
    }
    
    void MeetingSiteStorager::setActiveSite(SMART_PTR_NS::shared_ptr<MeetingSiteImpl> site, std::string sessionTicket, long keepAliveTime)
    {
        std::string siteUrl = site.get() == NULL? "" : site->getSiteUrl();
        
        CSFLogDebugS(jcfMeetingLogger, "MeetingSiteStorager::setActiveSite() siteUrl: " << siteUrl << "    sessionTicket: " << sessionTicket << "    keepAliveTime: " << keepAliveTime);
        
        std::map<std::string, SMART_PTR_NS::shared_ptr<MeetingSiteImpl> >::iterator it = m_mapAccountInfo.find(site->getSiteUrl());
        if (it != m_mapAccountInfo.end())
        {
            site->setEditableFlag(it->second->getEditableFlag());
            site->setDeletableFlag(it->second->getDeletableFlag());
            
            if (site->getUserName() == "")
            {
                site->setUserName(it->second->getUserName());
                site->setPassword(it->second->getPassword());
            }

            if (sessionTicket.length() > 0)
            {
                site->setSSOFlag(true);
            }
            else
            {
                //to solve Mac CSISecurePassIssue
                csf::SecureStringReader userPassReader;
                site->getPassword().getUnsecureString(userPassReader);
                if (userPassReader.c_str() == std::string("") || userPassReader.c_str() == std::string("********"))
                {
                    site->setPassword(it->second->getPassword());
                }
                
                site->setSSOFlag(it->second->getSSOFlag());
            }
        }
        else
        {
            site->setEditableFlag(true);
            site->setDeletableFlag(true);
            site->setSSOFlag(sessionTicket.length() > 0 ? true : false);
        }
    
		m_isRefreshSK = false;
        m_inProcessMeetingSite = site;
        m_inProcessSessionTicket = sessionTicket;
        m_inProcessKeepAliveTime = keepAliveTime;
        m_accountSync->setDefaultSite(site, sessionTicket, keepAliveTime);
    }
    
    void MeetingSiteStorager::deleteSite(std::string siteUrl)
    {
        CSFLogDebugS(jcfMeetingLogger, "MeetingSiteStorager::deleteSite() siteUrl: " << siteUrl);
        
        std::map<std::string, SMART_PTR_NS::shared_ptr<MeetingSiteImpl> >::iterator it = m_mapAccountInfo.find(siteUrl);
        if (it == m_mapAccountInfo.end())
        {
            MeetingAccountControllerImpl::getInstance()->fire_onMeetingSiteRemoved(siteUrl);
            return;
        }

        bool isDeleteDefault = (it->second->getSiteUrl() == m_defaultSiteUrl) ? true : false;
        m_mapAccountInfo.erase(it);
        m_accountSync->deleteSite(siteUrl);
        
        if (isDeleteDefault)
        {
            m_defaultSiteUrl = "";
            m_isActive = false;
            MeetingAccountSDKWrap::getInstance()->deleteDefaultAccount();
        }

        m_accountSync->updateAllSite(m_defaultSiteUrl, NULL, siteUrl);
		MeetingAccountControllerImpl::getInstance()->fire_onMeetingSiteRemoved(siteUrl);
	}
    
    void MeetingSiteStorager::checkSSOInfo(std::string siteUrl)
    {
        CSFLogDebugS(jcfMeetingLogger, "MeetingSiteStorager::checkSSOInfo()  siteUrl: " << siteUrl);
        
        setLastError(0);
        std::map<std::string, int>::iterator it = m_mapSSOInfo.find(siteUrl);
        if (it != m_mapSSOInfo.end())
        {
            MeetingAccountControllerImpl::getInstance()->fire_onSSOSiteChecked(siteUrl, it->second);
        }
        else
        {
            MeetingAccountSDKWrap::getInstance()->checkSSOSite(siteUrl);
        }
    }

	int MeetingSiteStorager::isSSOSite(std::string siteUrl)
	{
		return MeetingAccountSDKWrap::getInstance()->isSSOSite(siteUrl);
	}

	int MeetingSiteStorager::getSiteLastError()
	{
		return m_lastError;
	}

	std::string MeetingSiteStorager::getSiteLastErrorMessage()
	{
		return m_lastErrorMsg;
	}

	void MeetingSiteStorager::setLastError(int errorCode, std::string errorMsg)
	{
		m_lastError = errorCode;
		m_lastErrorMsg = errorMsg;
	}


    void MeetingSiteStorager::onVerifyAccount(bool isSuccess, int lastError, int siteFlag, std::string sessionTicket, long keepAliveTime)
    {
		if (m_isRefreshSK && !isSuccess)
		{
			m_isRefreshSK = false;
			CSFLogDebugS(jcfMeetingLogger, "MeetingSiteStorager::onVerifyAccount() refresh SDK failed will refresh again after one minute");
			return;
		}


        std::string originalSiteUrl = (NULL != m_inProcessMeetingSite.get()) ? m_inProcessMeetingSite->getSiteUrl() : "";
        
        CSFLogDebugS(jcfMeetingLogger, "MeetingSiteStorager::onVerifyAccount()  siteUrl: " << originalSiteUrl << "    isSuccess: " << isSuccess << "    lastError: " << lastError);
        
        csf::ScopedReadRWLock lock(mLock);
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = JabberServiceProvider::getInstance().getUnifiedFactory()->getServicesDispatcher();
        if (pServiceDispatcher != NULL)
            
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (!bOnDispatcher)
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&MeetingSiteStorager::onVerifyAccount, this, isSuccess, lastError, siteFlag, sessionTicket, keepAliveTime), "MeetingSiteStorager::onVerifyAccount");
                return;
            }
        }
        
		if (!isSuccess)
        {
            setReady();
			setLastError(lastError);
			
			MeetingAccountControllerImpl::getInstance()->fire_onMeetingSiteActived(false, originalSiteUrl, m_inProcessMeetingSite);
            return;
        }

        if (m_inProcessSessionTicket.length() == 0)
        {
            m_inProcessSessionTicket = sessionTicket;
            m_inProcessKeepAliveTime = keepAliveTime;
            
        }

        std::map<std::string, SMART_PTR_NS::shared_ptr<MeetingSiteImpl> >::iterator it = m_mapAccountInfo.find(m_inProcessMeetingSite->getSiteUrl());
        if (it == m_mapAccountInfo.end())
        {
            m_inProcessMeetingSite->setEditableFlag(true);
            m_inProcessMeetingSite->setDeletableFlag(true);
            m_inProcessMeetingSite->setDefaultSiteFlag(true);
            m_inProcessMeetingSite->setActiveFlag(true);
            m_inProcessMeetingSite->setSiteFlag(siteFlag);
            
            m_mapAccountInfo[m_inProcessMeetingSite->getSiteUrl()] = m_inProcessMeetingSite;
        }
        else
        {
            m_inProcessMeetingSite->setSiteDescription(it->second->getSiteDescription());
            m_inProcessMeetingSite->setEditableFlag(it->second->getEditableFlag());
            m_inProcessMeetingSite->setDeletableFlag(it->second->getDeletableFlag());
            m_inProcessMeetingSite->setDefaultSiteFlag(true);
            m_inProcessMeetingSite->setActiveFlag(true);
            m_inProcessMeetingSite->setSiteFlag(siteFlag);
            m_mapAccountInfo[m_inProcessMeetingSite->getSiteUrl()] = m_inProcessMeetingSite;
        }
        
        if (m_inProcessKeepAliveTime < 5400)
        {
            m_sessionTicketTimeout = getLocalTickCount() + 3600;
        }
        else
        {
            m_sessionTicketTimeout = getLocalTickCount() + m_inProcessKeepAliveTime - 3600;
        }
        m_isActive = true;
		m_isRefreshSK = false;
        m_defaultSiteUrl = m_inProcessMeetingSite->getSiteUrl();
        m_sessionTicket = m_inProcessSessionTicket;
        setReady();
		setLastError(0);
		MeetingAccountControllerImpl::getInstance()->fire_onMeetingSiteActived(true, m_inProcessMeetingSite->getSiteUrl(), m_inProcessMeetingSite);
        m_accountSync->updateAllSite(m_defaultSiteUrl, m_inProcessMeetingSite.get(), "");
        
        
    }
    
    void MeetingSiteStorager::onCheckSSOSite(int SSOFlag, std::string siteUrl)
    {
        CSFLogDebugS(jcfMeetingLogger, "MeetingSiteStorager::onCheckSSOSite()  siteUrl: " << siteUrl << "    SSOFlag: " << SSOFlag);
        
        csf::ScopedReadRWLock lock(mLock);
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = JabberServiceProvider::getInstance().getUnifiedFactory()->getServicesDispatcher();
        if (pServiceDispatcher != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (!bOnDispatcher)
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&MeetingSiteStorager::onCheckSSOSite, this, SSOFlag, siteUrl), "MeetingSiteStorager::onCheckSSOSite");
                return;
            }
        }
        
        setLastError(0);
        m_mapSSOInfo[siteUrl] = SSOFlag;
        MeetingAccountControllerImpl::getInstance()->fire_onSSOSiteChecked(siteUrl, (1 == SSOFlag) ? true : false);
    }
    
    void MeetingSiteStorager::onCheckSiteListSSOInfo()
    {
        CSFLogDebugS(jcfMeetingLogger, "MeetingSiteStorager::onCheckSiteListSSOInfo()");
        
        csf::ScopedReadRWLock lock(mLock);
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = JabberServiceProvider::getInstance().getUnifiedFactory()->getServicesDispatcher();
        if (pServiceDispatcher != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (!bOnDispatcher)
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&MeetingSiteStorager::onCheckSiteListSSOInfo, this), "MeetingSiteStorager::onCheckSiteListSSOInfo");
                return;
            }
        }

        std::map<std::string, SMART_PTR_NS::shared_ptr<MeetingSiteImpl> >::iterator it = m_mapAccountInfo.begin();
        while (it != m_mapAccountInfo.end())
        {
            std::string siteUrl = it->second->getSiteUrl();
            int ssoFlag = MeetingAccountSDKWrap::getInstance()->isSSOSite(siteUrl);
			it->second->setSSOFlag(ssoFlag);
			m_mapSSOInfo[siteUrl] = ssoFlag;
            it++;
        }
        
        setLastError(0);
        m_accountSync->getDefaultSite();
    }
    
    void MeetingSiteStorager::reset()
    {
        m_mapAccountInfo.clear();

        m_inProcessSessionTicket = "";
        m_inProcessKeepAliveTime = 0;
 
        m_isReady = false;
        m_isActive = false;
		m_isRefreshSK = false;
        m_defaultSiteUrl = "";
        m_sessionTicket = "";
        m_sessionTicketTimeout = 0;

        m_lastError = 0;
		m_lastErrorMsg = "";
    }
    
    long MeetingSiteStorager::getLocalTickCount()
    {
#ifdef WIN32
        return ::GetTickCount()/1000;
#else
        static uint64_t s_standardTick = mach_absolute_time();
        uint64_t currentTick = mach_absolute_time();
        
        mach_timebase_info_data_t timebase;
        mach_timebase_info(&timebase);
        
        return (double)(currentTick - s_standardTick) * (double)timebase.numer / (double)timebase.denom /1e9;
#endif
        
    }
}
